"use strict";
const config = require("config");
const $ = require("../gulp/config");
const m = require("../src/js/api");		// Lists all API Endpoints and a "getter" function to retrieve the URL's of the endpoints in use

const AVCodes = require("./AVCodes");	// constants file containing all constants used in tests (like Access/Verify Codes, etc)
const schema = require("./schemas/api").definitions.Facility;

const authURL = m.getURL("Authenticate");	// URL for authenticating
const testURL = m.getURL("FacilitiesDetails");	// URL for the facility endpoint
const TestName = "Test Facility Information Endpoint";

/**
 *	This test covers the facilities details endpoint and performs the following operations
 *		1. Authenticate and receive a JWT for authentication
 *		2. Issue a request witout a valid JWT - This should receive an error
 *		3. Issue a request WITH a valid JWT and no station ID - This should receive an array JSON objects which represents all the facilities
 *		4. Issue a request WITH a valid JWT and an INvalid station ID - This should receive a blank message in response
 *		5. Issue a request WITH a valid JWT and a valid station ID - This should receive a JSON Object containing the detailed information for the specified station ID
 *
 *	Details of the endpoints tested here can be found in: https://github.com/DBITPRO/VistA-Surgery-API/blob/master/Schema/VistASurgeryAPI.json
 **/

/**
 *	Note: CHAI References -
 *		Assertions - http://chaijs.com/api/assert/
 *		Expect/Should - http://chaijs.com/api/bdd/
 **/

describe(TestName, function(){
	/**
	 *	This function must be the first function as it gets a required JSON Web Token which is needed to authenticate further communication with the API
	 *	Each test file is designed to be a stand alone test.
	 **/

	let subTestName = "";
	let token = "";
	let body = "";
	let facility = "";

	it("Passing valid credentials should get a a valid JWT from the server from the server", function(done){
		let request = $.request("");
		request.post(authURL)
		.set("Content-Type", "application/x-www-form-urlencoded")
		.send(AVCodes.GOOD_AVCodes)
		.end(function(err, res){
			$.expect(err).to.equal(null, "We received an error -", JSON.stringify(err));
			$.expect(res.statusCode).to.equal(200, `Received an unexpected StatusCode (${res.statusCode}) from the server`);
			$.expect(res.body.message).to.equal("Enjoy your token!", "Invalid Message - ");
			$.expect(res.body.token).to.not.equal("", "No Token Provided");
			body = res.body;
			token = body.token;
			done();
		});
	});

	it("Requesting Station Information without a valid token should produce an error", function(done) {
		let request = $.request("");
		request.get(testURL)
		.end(function(err, res){
			$.expect(err).to.equal(null, "We received an error -", JSON.stringify(err));
			$.expect(res.statusCode).to.equal(401, `Received an unexpected StatusCode (${res.statusCode}) from the server`);
			$.expect(res.text).to.equal("No token provided.", "Invalid Message - ");
			done();
		});
	});

	it("Requesting Station Information WITH a valid token should return an array of facility data", function(done) {
		let request = $.request("");
		request.get(testURL)
		.set("Authorization", token)
		.end(function(err, res){
			$.expect(err).to.equal(null, "We received an error -", JSON.stringify(err));
			$.expect(res.statusCode).to.equal(200, `Received an unexpected StatusCode (${res.statusCode}) from the server`);
			$.assert.isArray(res.body, "We did not receive an array of facilities");
			$.assert.isObject(res.body[0], "The array of facilities did not contain a valid facility object -", JSON.stringify(res.body[0]))
			done();
		});
	});

	it("Requesting Station Information WITH a valid token and an INvalid facility ID should return an empty message", function(done) {
		let request = $.request("");
		let BadFacility = testURL + "/XXX";
		request.get(BadFacility)
		.set("Authorization", token)
		.end(function(err, res){
			$.expect(err).to.equal(null, "We received an error -", JSON.stringify(err));
			$.expect(res.statusCode).to.equal(200, `Received an unexpected StatusCode (${res.statusCode}) from the server`);
			$.expect(res.text, "Invalid Message - ").to.equal("");
			$.expect(res.body, "Invalid Response - ").to.equal("");
			done();
		});
	});

	it("Requesting Station Information WITH a valid token and a valid facility ID should return facility data", function(done) {
		let request = $.request("");
		let GoodFacility = testURL + "/" + AVCodes.Good_StationID;
		request.get(GoodFacility)
		.set("Content-Type", "application/x-www-form-urlencoded")
		.set("Authorization", token)
		.end(function(err, res){
			$.expect(err).to.equal(null, "We received an error -", JSON.stringify(err));
			$.expect(res.statusCode).to.equal(200, `Received an unexpected StatusCode (${res.statusCode}) from the server`);
			$.assert.isObject(res.body, "facilities did not contain a valid facility object -", JSON.stringify(res.body));
			$.expect(res.body.stationID).to.equal(AVCodes.Good_StationID, "Received invalid Facility Information - ", JSON.stringify(res.body));
			facility = res.body;
			done();
		});
	});
	it("Facility Information received should match data schema", function(done) {
		$.expect(facility).to.be.jsonSchema(schema);
		done();
	});
});
